import { rmSync, readdir, existsSync } from 'fs'
import { join } from 'path'
import pino from 'pino'
import makeWASocketModule, {
    useMultiFileAuthState,
    makeCacheableSignalKeyStore,
    DisconnectReason,
    delay,
    downloadMediaMessage,
    getAggregateVotesInPollMessage,
    fetchLatestBaileysVersion,
    WAMessageStatus,
} from 'baileys'

import proto from 'baileys'

import makeInMemoryStore from './store/memory-store.js'
import makeDatabaseStore from './store/database-store.js'
import { useDatabaseAuthState } from './store/database-auth-state.js'

import { toDataURL } from 'qrcode'
import __dirname from './dirname.js'
import response from './response.js'
import { downloadImage } from './utils/download.js'
import axios from 'axios'
import NodeCache from 'node-cache'
import prisma from './utils/prisma.js'
import logger from './utils/logger.js'

const msgRetryCounterCache = new NodeCache()

const sessions = new Map()
const retries = new Map()

const APP_WEBHOOK_ALLOWED_EVENTS = process.env.APP_WEBHOOK_ALLOWED_EVENTS.split(',')
const STORE_TYPE = process.env.STORE_TYPE || 'file'

const sessionsDir = (sessionId = '') => {
    return join(__dirname, 'sessions', sessionId ? sessionId : '')
}

const isSessionExists = (sessionId) => {
    return sessions.has(sessionId)
}

const isSessionConnected = (sessionId) => {
    return sessions.get(sessionId)?.ws?.socket?.readyState === 1
}

const shouldReconnect = (sessionId) => {
    const maxRetries = parseInt(process.env.MAX_RETRIES ?? 0)
    let attempts = retries.get(sessionId) ?? 0

    // MaxRetries = maxRetries < 1 ? 1 : maxRetries
    if (attempts < maxRetries || maxRetries === -1) {
        ++attempts
        logger.info('Reconnecting...', { attempts, sessionId })
        retries.set(sessionId, attempts)

        return true
    }

    return false
}





/**
 * @returns {(import('baileys').AnyWASocket|null)}
 */
const getSession = (sessionId) => {
    return sessions.get(sessionId) ?? null
}

const getListSessions = () => {
    return [...sessions.keys()]
}

const deleteSession = async (sessionId) => {
    const sessionFile = 'md_' + sessionId
    const storeFile = `${sessionId}_store.json`
    const rmOptions = { force: true, recursive: true }

    if (STORE_TYPE === 'file') {
        rmSync(sessionsDir(sessionFile), rmOptions)
        rmSync(sessionsDir(storeFile), rmOptions)
    } else {
        // Delete from DB
        try {
            await prisma.session.deleteMany({
                where: { sessionId }
            })
            await prisma.chat.deleteMany({
                where: { sessionId }
            })
            await prisma.contact.deleteMany({
                where: { sessionId }
            })
            await prisma.message.deleteMany({
                where: { sessionId }
            })
        } catch (error) {
            logger.error('Error deleting session from DB:', error)
        }
    }

    sessions.delete(sessionId)
    retries.delete(sessionId)
}

const getChatList = async (sessionId, isGroup = false, limit = 20, cursor = null) => {
    const filter = isGroup ? '@g.us' : '@s.whatsapp.net'
    let chats = []

    if (STORE_TYPE === 'file') {
        chats = [...getSession(sessionId).store.chats.values()]
        chats = chats.filter(chat => chat.id.endsWith(filter))

        // Simple pagination for file store
        const startIndex = cursor ? parseInt(cursor) : 0
        const paginatedChats = chats.slice(startIndex, startIndex + limit)
        const hasMore = startIndex + limit < chats.length

        return {
            chats: paginatedChats,
            nextCursor: hasMore ? (startIndex + limit).toString() : null,
            hasMore
        }
    } else {
        const result = await getSession(sessionId).store.chats.getAll(limit, cursor, filter)

        // Convert BigInt to string for JSON serialization
        result.chats = result.chats.map(chat => ({
            ...chat,
            conversationTimestamp: chat.conversationTimestamp ? Number(chat.conversationTimestamp) : null,
            ephemeralSettingTimestamp: chat.ephemeralSettingTimestamp ? Number(chat.ephemeralSettingTimestamp) : null,
            lastMsgTimestamp: chat.lastMsgTimestamp ? Number(chat.lastMsgTimestamp) : null,
            muteEndTime: chat.muteEndTime ? Number(chat.muteEndTime) : null,
            lastUpdated: chat.lastUpdated ? Number(chat.lastUpdated) : null
        }))

        return result
    }
}

/**
 * @param {import('baileys').AnyWASocket} session
 */
const isExists = async (session, jid, isGroup = false) => {
    try {
        let result

        if (isGroup) {
            result = await session.groupMetadata(jid)

            return Boolean(result.id)
        }

        ;[result] = await session.onWhatsApp(jid)

        return result.exists
    } catch {
        return false
    }
}

/**
 * @param {import('baileys').AnyWASocket} session
 */
const sendMessage = async (session, receiver, message, options = {}, delayMs = 1000) => {
    try {
        await delay(parseInt(delayMs))
        return await session.sendMessage(receiver, message, options)
    } catch {
        return Promise.reject(null) // eslint-disable-line prefer-promise-reject-errors
    }
}

/**
 * @param {import('baileys').AnyWASocket} session
 */
const updateProfileStatus = async (session, status) => {
    try {
        return await session.updateProfileStatus(status)
    } catch {
        return Promise.reject(null) // eslint-disable-line prefer-promise-reject-errors
    }
}

const updateProfileName = async (session, name) => {
    try {
        return await session.updateProfileName(name)
    } catch {
        return Promise.reject(null) // eslint-disable-line prefer-promise-reject-errors
    }
}

const getProfilePicture = async (session, jid, type = 'image') => {
    try {
        return await session.profilePictureUrl(jid, type)
    } catch {
        return Promise.reject(null) // eslint-disable-line prefer-promise-reject-errors
    }
}

const blockAndUnblockUser = async (session, jid, block) => {
    try {
        return await session.updateBlockStatus(jid, block)
    } catch {
        return Promise.reject(null) // eslint-disable-line prefer-promise-reject-errors
    }
}

const formatPhone = (phone) => {
    if (phone.endsWith('@s.whatsapp.net')) {
        return phone
    }

    let formatted = phone.replace(/\D/g, '')

    return (formatted += '@s.whatsapp.net')
}

const formatGroup = (group) => {
    if (group.endsWith('@g.us')) {
        return group
    }

    let formatted = group.replace(/[^\d-]/g, '')

    return (formatted += '@g.us')
}

const cleanup = () => {
    logger.info('Running cleanup before exit.')

    sessions.forEach((session, sessionId) => {
        if (STORE_TYPE === 'file') {
            session.store.writeToFile(sessionsDir(`${sessionId}_store.json`))
        }
    })
}

const getGroupsWithParticipants = async (session) => {
    return session.groupFetchAllParticipating()
}

const participantsUpdate = async (session, jid, participants, action) => {
    return session.groupParticipantsUpdate(jid, participants, action)
}

const updateSubject = async (session, jid, subject) => {
    return session.groupUpdateSubject(jid, subject)
}

const updateDescription = async (session, jid, description) => {
    return session.groupUpdateDescription(jid, description)
}

const settingUpdate = async (session, jid, settings) => {
    return session.groupSettingUpdate(jid, settings)
}

const leave = async (session, jid) => {
    return session.groupLeave(jid)
}

const inviteCode = async (session, jid) => {
    return session.groupInviteCode(jid)
}

const revokeInvite = async (session, jid) => {
    return session.groupRevokeInvite(jid)
}

const metaData = async (session, req) => {
    return session.groupMetadata(req.groupId)
}

const acceptInvite = async (session, req) => {
    return session.groupAcceptInvite(req.invite)
}

const profilePicture = async (session, jid, urlImage) => {
    const image = await downloadImage(urlImage)
    return session.updateProfilePicture(jid, { url: image })
}

const readMessage = async (session, keys) => {
    return session.readMessages(keys)
}

const readConversation = async (sessionId, jid) => {
    try {
        const session = getSession(sessionId)

        // Load recent messages from the chat
        const messages = await session.store.loadMessages(jid, 100, null)

        // Filter unread messages (messages not from me)
        const unreadMessages = messages.filter(msg => !msg.key.fromMe)

        if (unreadMessages.length > 0) {
            // Mark all unread messages as read
            const keys = unreadMessages.map(msg => msg.key)
            await session.readMessages(keys)
        }

        // Update unreadCount in database if using database store
        if (STORE_TYPE === 'database') {
            await prisma.chat.updateMany({
                where: {
                    sessionId,
                    id: jid
                },
                data: {
                    unreadCount: 0
                }
            })
        }

        return { success: true, markedCount: unreadMessages.length }
    } catch (error) {
        logger.error('Error marking conversation as read:', error)
        throw error
    }
}

const getStoreMessage = async (session, messageId, remoteJid) => {
    try {
        // Load messages from the chat
        const messages = await session.store.loadMessages(remoteJid, 100, null)

        // Find the specific message by ID
        const message = messages.find(msg => msg.key.id === messageId)

        if (!message) {
            throw new Error('Message not found')
        }

        return message
    } catch {
        // eslint-disable-next-line prefer-promise-reject-errors
        return Promise.reject(null)
    }
}

const getMessageMedia = async (session, message) => {
    try {
        const messageType = Object.keys(message.message)[0]
        const mediaMessage = message.message[messageType]
        const buffer = await downloadMediaMessage(
            message,
            'buffer',
            {},
            { reuploadRequest: session.updateMediaMessage },
        )

        return {
            messageType,
            fileName: mediaMessage.fileName ?? '',
            caption: mediaMessage.caption ?? '',
            size: {
                fileLength: mediaMessage.fileLength,
                height: mediaMessage.height ?? 0,
                width: mediaMessage.width ?? 0,
            },
            mimetype: mediaMessage.mimetype,
            base64: buffer.toString('base64'),
        }
    } catch {
        // eslint-disable-next-line prefer-promise-reject-errors
        return Promise.reject(null)
    }
}

const getMessageBuffer = async (session, message) => {
    try {
        const messageType = Object.keys(message.message)[0]
        const mediaMessage = message.message[messageType]
        const buffer = await downloadMediaMessage(
            message,
            'buffer',
            {},
            { reuploadRequest: session.updateMediaMessage },
        )

        return {
            buffer,
            fileName: mediaMessage.fileName ?? 'file',
            mimetype: mediaMessage.mimetype,
        }
    } catch {
        // eslint-disable-next-line prefer-promise-reject-errors
        return Promise.reject(null)
    }
}

const convertToBase64 = (arrayBytes) => {
    const byteArray = new Uint8Array(arrayBytes)
    return Buffer.from(byteArray).toString('base64')
}

const createSession = () =>{
    return true;
}

const init = async () => {
    if (STORE_TYPE === 'file') {
        readdir(sessionsDir(), (err, files) => {
            if (err) {
                throw err
            }

            for (const file of files) {
                if ((!file.startsWith('md_') && !file.startsWith('legacy_')) || file.endsWith('_store')) {
                    continue
                }

                const filename = file.replace('.json', '')
                const sessionId = filename.substring(3)
                logger.info('Recovering session: ' + sessionId)
              
            }
        })
    } else {
        // Recover from DB
        const sessions = await prisma.session.findMany({
            select: { sessionId: true },
            distinct: ['sessionId']
        })

        for (const { sessionId } of sessions) {
            logger.info('Recovering session from DB: ' + sessionId)
          
        }
    }
}

export {
    isSessionExists,
    getSession,
    getListSessions,
    deleteSession,
    getChatList,
    getGroupsWithParticipants,
    isExists,
    sendMessage,
    updateProfileStatus,
    updateProfileName,
    getProfilePicture,
    formatPhone,
    formatGroup,
    cleanup,
    participantsUpdate,
    updateSubject,
    updateDescription,
    settingUpdate,
    leave,
    inviteCode,
    revokeInvite,
    metaData,
    acceptInvite,
    profilePicture,
    readMessage,
    readConversation,
    init,
    createSession,
    isSessionConnected,
    getMessageMedia,
    getMessageBuffer,
    getStoreMessage,
    blockAndUnblockUser,
}