<?php

namespace Modules\WhatsappWeb\App\Jobs;

use App\Models\CampaignLog;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class UpdateMessageStatusJob implements ShouldQueue
{
    use Dispatchable;

    /**
     * Create a new job instance.
     */
    public function __construct(public array $payload)
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $event = data_get($this->payload, 'event');

        $messageId = $this->getMessageId();

        if (! $messageId) {
            logOnDebug('No message id found', $this->payload);

            return;
        }

        $log = CampaignLog::query()
            ->where('module', 'whatsapp-web')
            ->where('meta->response->key->id', $messageId)->first();

        if (! $log) {
            logOnDebug('No log found', $this->payload);

            return;
        }

        $updateColumns = match ($event) {
            'MESSAGES_UPDATE' => $this->getUpdateColumnsForMessagesUpdate($log),
            default => [],
        };

        if (! $updateColumns) {
            logOnDebug('No update columns found', $this->payload);

            return;
        }

        $log->update($updateColumns);
    }

    private function getUpdateColumnsForMessagesUpdate(CampaignLog $log): array
    {
        $messageId = data_get($this->payload, 'data.messages.key.id');
        $status = data_get($this->payload, 'data.messages.status');

        if (! $messageId || ! $status) {
            logOnDebug('No message id or status found', $this->payload);

            return [];
        }

        $updateColumn = match ($status) {
            2 => 'send_at',
            3 => 'delivered_at',
            4 => 'read_at',
            default => null
        };

        if (! $updateColumn) {
            logOnDebug('No update column found', $this->payload);

            return [];
        }

        $updateColumns = [$updateColumn => now()];

        if ($log->send_at === null && in_array($updateColumn, ['delivered_at', 'read_at'])) {
            $updateColumns['send_at'] = now();
        } else {
            logOnDebug('No update column found', $this->payload);
        }

        return $updateColumns;
    }

    private function getMessageId(): string
    {
        return data_get($this->payload, 'data.messages.key.id');
    }
}
