<?php

namespace Modules\Whatsapp\App\Services;

use App\Models\Campaign;
use App\Models\CampaignLog;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\Template;
use App\Models\Template as TemplateModel;
use Illuminate\Support\Facades\Log;

class CampaignService
{
    /**
     * Convert internal message types to WhatsApp API compatible types (lowercase)
     */
    private static function convertMessageTypeForApi(string $messageType): string
    {
        $typeMap = [
            'text' => 'text',
            'image' => 'image',
            'audio' => 'audio',
            'video' => 'video',
            'document' => 'document',
            'location' => 'location',
            'contact' => 'contacts',
            'interactive' => 'interactive',
            'cloud_template' => 'template',
            'custom_template' => 'template',
            'template' => 'template',
        ];

        return $typeMap[$messageType] ?? strtolower($messageType);
    }

    /**
     * Extract proper message body from template based on its type
     * Returns WhatsApp API compatible message structure
     */
    private static function extractMessageBodyFromTemplate(TemplateModel $template): array
    {
        $meta = $template->meta ?? [];
        $templateType = $template->type;

        switch ($templateType) {
            case 'text':
                return [
                    'body' => $meta['body'] ?? ''  // MessageService expects 'body', payload will use 'text' key
                ];

            case 'image':
                return [
                    'link' => $meta['link'] ?? '',
                    'caption' => $meta['caption'] ?? ''
                ];

            case 'audio':
                return [
                    'link' => $meta['link'] ?? ''
                ];

            case 'video':
                return [
                    'link' => $meta['link'] ?? '',
                    'caption' => $meta['caption'] ?? ''
                ];

            case 'document':
                return [
                    'link' => $meta['link'] ?? '',
                    'caption' => $meta['caption'] ?? ''
                ];

            case 'location':
                return [
                    'latitude' => $meta['latitude'] ?? '',
                    'longitude' => $meta['longitude'] ?? '',
                    'name' => $meta['name'] ?? '',
                    'address' => $meta['address'] ?? ''
                ];

            case 'contact':
                return $meta;

            default:
                // Fallback for unknown types - return as text
                return [
                    'body' => is_string($meta) ? $meta : json_encode($meta)
                ];
        }
    }

    public static function send(Campaign $campaign)
    {
        Log::debug('CampaignService::send called with campaign:', [
            'campaign_id' => $campaign->id,
            'campaign_message_type' => $campaign->message_type,
            'campaign_template_id' => $campaign->template_id,
            'campaign_meta_keys' => array_keys($campaign->meta ?? []),
        ]);

        $customers = $campaign->group?->customers ?? [];

        ini_set('max_execution_time', count($customers) * 5);

        foreach ($customers as $customer) {
            sleep(1);
            $conversation = Conversation::firstOrCreate([
                'module' => 'whatsapp',
                'platform_id' => $campaign->platform_id,
                'owner_id' => $campaign->owner_id,
                'customer_id' => $customer->id,
            ], [
                'badge_id' => null,
                'auto_reply_enabled' => $campaign->platform->isAutoReplyEnabled(),
                'meta' => [],
            ]);

            if (in_array($campaign->message_type, ['template', 'cloud_template', 'custom_template', 'interactive'])) {
                // For custom templates, treat them as regular messages (not template messages)
                if ($campaign->message_type === 'custom_template' && $campaign->template_id) {
                    $originalTemplate = TemplateModel::where('id', $campaign->template_id)->first();
                    
                    Log::debug('Processing custom template campaign:', [
                        'campaign_id' => $campaign->id,
                        'campaign_message_type' => $campaign->message_type,
                        'campaign_template_id' => $campaign->template_id,
                        'template_found' => $originalTemplate ? 'yes' : 'no',
                        'template_type' => $originalTemplate->type ?? 'null',
                        'template_meta_keys' => $originalTemplate ? array_keys($originalTemplate->meta ?? []) : 'null',
                    ]);
                    
                    if ($originalTemplate) {
                        // Extract the proper message body based on template type
                        $messageBody = self::extractMessageBodyFromTemplate($originalTemplate);
                        
                        $message = new Message([
                            'module' => 'whatsapp',
                            'uuid' => null,
                            'conversation_id' => $conversation->id,
                            'platform_id' => $conversation->platform_id,
                            'customer_id' => $conversation->customer_id,
                            'owner_id' => $conversation->owner_id,
                            'direction' => 'out',
                            'type' => self::convertMessageTypeForApi($originalTemplate->type),
                            'body' => $messageBody,
                        ]);
                        
                        Log::debug('Created message from custom template:', [
                            'message_type' => $message->type,
                            'message_body_keys' => array_keys($message->body ?? []),
                        ]);
                    } else {
                        Log::error('Custom template not found for campaign:', [
                            'campaign_id' => $campaign->id,
                            'template_id' => $campaign->template_id,
                        ]);
                        throw new \Exception('Custom template not found');
                    }
                } else {
                    // For cloud templates, load the actual template from database
                    if ($campaign->message_type === 'cloud_template' && $campaign->template_id) {
                        $originalTemplate = TemplateModel::where('id', $campaign->template_id)->first();
                        if ($originalTemplate) {
                            Log::debug('Processing cloud template campaign:', [
                                'campaign_id' => $campaign->id,
                                'template_id' => $originalTemplate->id,
                                'template_name' => $originalTemplate->name,
                                'template_type' => $originalTemplate->type,
                            ]);
                            
                            $templateService = new TemplateService($originalTemplate, $conversation, $customer);
                            $message = $templateService->generateMessage($conversation, $customer);
                        } else {
                            Log::error('Cloud template not found for campaign:', [
                                'campaign_id' => $campaign->id,
                                'template_id' => $campaign->template_id,
                            ]);
                            throw new \Exception('Cloud template not found');
                        }
                    } else {
                        // For other template types (template, interactive)
                        $templateData = [
                            'module' => $campaign->module,
                            'owner_id' => $campaign->owner_id,
                            'name' => data_get($campaign, 'meta.name'),
                            'meta' => $campaign->meta,
                            'type' => self::convertMessageTypeForApi($campaign->message_type),
                        ];

                        $template = new Template($templateData);
                        $templateService = new TemplateService($template);
                        $message = $templateService->generateMessage($conversation, $customer);
                    }
                }
            } else {
                $message = new Message([
                    'module' => 'whatsapp',
                    'uuid' => null,
                    'conversation_id' => $conversation->id,
                    'platform_id' => $conversation->platform_id,
                    'customer_id' => $conversation->customer_id,
                    'owner_id' => $conversation->owner_id,
                    'direction' => 'out',
                    'type' => self::convertMessageTypeForApi($campaign->message_type),
                    'body' => $campaign->meta,
                ]);
            }

            $messageService = new MessageService($message);
            $sendMessage = $messageService->send();

            CampaignLog::create([
                'module' => 'whatsapp',
                'owner_id' => $conversation->owner_id,
                'campaign_id' => $campaign->id,
                'message_id' => $sendMessage->id,
                'customer_id' => $customer->id,
                'send_at' => now(),
                'meta' => [
                    'phone' => $customer->uuid,
                    'wamid' => $sendMessage->uuid,
                ],
            ]);

        }

        $campaign->update(['status' => Campaign::STATUS_SEND]);
    }
}
