<?php

namespace Modules\Whatsapp\App\Http\Controllers;

use App\Helpers\PageHeader;
use App\Helpers\PlanPerks;
use App\Http\Controllers\Controller;
use App\Models\Campaign;
use App\Models\Group;
use App\Models\Template;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Modules\Whatsapp\App\Services\CampaignService;
use Modules\Whatsapp\App\Services\TemplateValidation;

class CampaignController extends Controller
{
    public $validation_params = [
        'name' => 'required|string|max:191',
        'platform_id' => 'required|exists:platforms,id',
        'group_id' => 'required|numeric|exists:groups,id',
        'message_type' => 'required|in:text,image,audio,video,document,location,contact,interactive,template,cloud_template,custom_template',
        'template_id' => 'required_if:message_type,template,cloud_template,custom_template',
        'message' => 'nullable',
        'send_type' => 'required|string|in:instant,draft,scheduled',
        'meta' => 'array',
    ];

    public $validation_message = [
        'name.required' => 'The name field is required',
        'platform_id.required' => 'The device field is required',
        'group_id.required' => 'The Group field is required',
        'message_type.required' => 'The message type field is required',
        'template_id.required' => 'The Template field is required',
        'message.required' => 'The message field is required when message type is text.',
        'meta.type.required' => 'The Interactive type is required',
        'meta.params' => 'The Interactive Params is required ',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = activeWorkspaceOwner()->campaigns()->whatsapp();

        $campaigns = $query->clone()
            ->filterOn(['name', 'status', 'message_type'])
            ->with(['group:id,name', 'template:id,name'])
            ->latest()
            ->paginate();

        PageHeader::set()->title('Campaigns')
            ->addLink('New Campaign', route('user.whatsapp.campaigns.create'), 'bx:plus')
            ->overviews([
                [
                    'icon' => 'bx:grid-alt',
                    'title' => 'Total  Campaigns',
                    'value' => $campaigns->total(),
                ],
                [
                    'icon' => 'bx:file',
                    'title' => 'Draft Campaigns',
                    'value' => $query->clone()->whereStatus(Campaign::STATUS_PENDING)->count(),
                ],
                [
                    'icon' => 'bx:timer',
                    'title' => 'Scheduled Campaigns',
                    'value' => $query->clone()->whereStatus(Campaign::STATUS_SCHEDULED)->count(),
                ],
                [
                    'icon' => 'bx:check-circle',
                    'title' => 'Send Campaigns',
                    'value' => $query->clone()->whereStatus(Campaign::STATUS_SEND)->count(),
                ],
            ]);
        $systemTimezone = env('TIME_ZONE', 'UTC');

        return Inertia::render('Campaigns/Index', [
            'campaigns' => $campaigns,
            'systemTimezone' => $systemTimezone,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {

        validateWorkspacePlan('campaign');

        PageHeader::set()->title('Campaign Create')->buttons([
            [
                'url' => route('user.whatsapp.campaigns.index'),
                'text' => 'Back',
            ],
        ]);

        $user = activeWorkspaceOwner();

        $groups = $user->groups()
            ->whatsapp()
            ->select('id', 'name')
            ->latest()
            ->get();

        $platforms = $user->platforms()
            ->whatsapp()
            ->select('id', 'name')
            ->latest()
            ->get();

        // Get custom templates for initial load (exclude interactive, cloud_template, and WhatsApp cloud templates)
        $customTemplates = \App\Models\Template::where('module', 'whatsapp')
            ->where('owner_id', $user->id)
            ->whereNotIn('type', ['interactive', 'cloud_template', 'template'])
            ->select('id', 'name', 'type', 'meta', 'status')
            ->get();
            
        // Debug: Log all template types to see what's in the database
        \Log::debug('All user templates:', \App\Models\Template::where('module', 'whatsapp')
            ->where('owner_id', $user->id)
            ->select('id', 'name', 'type')
            ->get()->toArray());
            
        \Log::debug('Filtered custom templates:', $customTemplates->toArray());

        $time_zone_list = timezone_identifiers_list();

        return Inertia::render('Campaigns/Create', compact('groups', 'platforms', 'time_zone_list', 'customTemplates'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, ?Campaign $editCampaign = null)
    {
          if(env('DEMO_MODE') && auth()->user()->id == 3){
            return back()->with('danger', __('Permission disabled for demo account please create a test account..!'));
        }
        validateWorkspacePlan('campaign');
        validateWorkspacePlan('cloud_messages');

        $user = activeWorkspaceOwner();

        $messageLimit = data_get($user->plan_data, 'cloud_messages.value', 0);

        $isUnlimited = $messageLimit === -1;
        if (! $isUnlimited) {
            $monthCycle = PlanPerks::calculateCurrentCycleUsage($user);
            $alreadySendMessageCount = $user->messages()
                ->whatsapp()
                ->whereBetween('created_at', [$monthCycle['start'], $monthCycle['end']])
                ->count();

            $remainingMessageCount = $messageLimit - $alreadySendMessageCount;
            $newMessageCount = $user->groups()->whatsapp()->find($request->group_id)?->customers()->count();

            if ($newMessageCount > $remainingMessageCount) {
                return back()->with('danger', 'Message limit exceeded. Please upgrade your plan to send more messages.');
            }
        }

        $validated = $request->validate($this->validation_params);

        // Custom validation for text messages
        if ($request->message_type === 'text') {
            $textContent = $request->message ?? data_get($request->meta, 'body', '');
            if (empty(trim($textContent))) {
                return back()->withErrors(['message' => 'The message field is required when message type is text.'])->withInput();
            }
        }

        if ($request->send_type == 'scheduled') {

            $request->validate([
                'schedule_time' => 'required|date|after:now',
                'timezone' => 'required|timezone',
            ]);
            $validated['schedule_time'] = $request->schedule_time;
            $validated['timezone'] = $request->timezone;
        }

        $isTemplateMessage = in_array($request->message_type, ['template', 'cloud_template', 'custom_template', 'interactive']);

        if ($isTemplateMessage) {
            TemplateValidation::validate($request);
        }

        if (isset($validated['schedule_time']) && isset($validated['timezone'])) {
            $schedule_at = Carbon::parse($request->schedule_time);
            $dateTime = Carbon::createFromFormat('Y-m-d H:i:s', $schedule_at, $request->timezone);
            $dateTime = $dateTime->copy()->tz(env('TIME_ZONE', 'UTC'));

            $validated['schedule_at'] = $dateTime;
            $validated['timezone'] = $request->timezone;
            $validated['send_type'] = 'scheduled';

            $validated['meta']['schedule_timezone'] = $request->timezone;
            $validated['meta']['schedule_timestamp'] = $request->schedule_time;
        }

        $user = activeWorkspaceOwner();
        DB::beginTransaction();
        if ($request->get('message_type') == 'interactive' && $request->get('save_as_template')) {
            $template = Template::create([
                'module' => 'whatsapp',
                'platform_id' => $request->platform_id,
                'owner_id' => $user->id,
                'name' => $request->name,
                'type' => 'interactive',
                'meta' => $request->meta,
            ]);
            $validated['template_id'] = $template->id;
        }

        // Handle message type processing
        if ($request->message_type === 'text') {
            $messageText = $request->message ?? data_get($request->meta, 'body', '');
            $validated['meta'] = [
                ...$validated['meta'],
                'type' => 'text',
                'body' => $messageText,
                'text' => $messageText,
            ];
        } elseif ($request->message_type === 'custom_template') {
            // For custom templates, ensure the meta contains the original template type
            if ($request->template_id) {
                $template = Template::find($request->template_id);
                if ($template) {
                    $validated['meta'] = [
                        ...$validated['meta'],
                        'type' => $template->type, // This should be the original message type (text, image, etc.)
                        'original_template_id' => $template->id,
                        'original_template_name' => $template->name,
                    ];
                }
            }
        }

        $validated['module'] = 'whatsapp';
        $campaign = $user->campaigns()->updateOrCreate(
            [
                'id' => $editCampaign?->id ?? null,
            ],
            $validated
        );

        switch ($campaign->send_type) {
            case 'scheduled':
                $campaign->update([
                    'status' => Campaign::STATUS_SCHEDULED,
                ]);
                $message = 'Campaign Scheduled Successfully';
                break;
            case 'draft':
                $message = 'Campaign Draft Saved Successfully';
                break;
        }

        if ($campaign->send_type == 'instant') {
            CampaignService::send($campaign);
            $message = 'Campaign Sent Successfully';
        }

        DB::commit();

        return to_route('user.whatsapp.campaigns.index')
            ->with('success', $message);
    }

    public function show(Campaign $campaign)
    {

        $campaign = $campaign->loadCount([
            'logs as total_messages',
            'logs as failed_messages' => fn ($q) => $q->whereNotNull('failed_at'),
            'logs as delivered_messages' => fn ($q) => $q->whereNotNull('delivered_at'),
            'logs as sent_messages' => fn ($q) => $q->whereNotNull('send_at'),
            'logs as read_messages' => fn ($q) => $q->whereNotNull('read_at'),
        ]);
        $query = $campaign->logs();

        $overviews = [
            [
                'icon' => 'bx:grid-alt',
                'value' => $query->clone()->count(),
                'title' => 'Total',
            ],
            [
                'icon' => 'bx:grid-alt',
                'value' => $query->clone()->whereNotNull('send_at')->count(),
                'title' => 'Send',
            ],
            [
                'icon' => 'bx-check-circle',
                'value' => $query->clone()->whereNotNull('delivered_at')->count(),
                'title' => 'Delivered',
            ],
            [
                'icon' => 'bx:timer',
                'value' => $query->clone()->whereNotNull('read_at')->count(),
                'title' => 'Read',
            ],
        ];
        PageHeader::set(__('Campaign Logs'))->addBackLink(route('user.whatsapp.campaigns.index'))->overviews($overviews);

        $logs = $campaign->logs()
            ->when(request()->filled('search'), function ($q) {
                $q
                    ->where('waid', 'LIKE', '%'.request('search').'%')
                    ->orWhere('wamid', 'LIKE', '%'.request('search').'%');
            })

            ->paginate();

        return Inertia::render('Campaigns/Logs', [
            'logs' => $logs,
            'campaign' => $campaign,
        ]);
    }

    public function edit(Campaign $campaign)
    {
        PageHeader::set()->title('Campaign Edit')->buttons([
            [
                'url' => route('user.whatsapp.campaigns.index'),
                'text' => 'Back',
            ],
        ]);

        $groups = Group::select('id', 'name')
            ->whatsapp()
            ->latest()
            ->get();

        /** @var \App\Models\User */
        $user = activeWorkspaceOwner();
        $platforms = $user->platforms()
            ->whatsapp()
            ->select('id', 'name')
            ->latest()
            ->get();

        // Get custom templates for edit page as well
        $customTemplates = \App\Models\Template::where('module', 'whatsapp')
            ->where('owner_id', $user->id)
            ->whereNotIn('type', ['interactive', 'cloud_template'])
            ->select('id', 'name', 'type', 'meta', 'status')
            ->get();

        $time_zone_list = timezone_identifiers_list();

        return Inertia::render('Campaigns/Create', [
            'editCampaign' => $campaign,
            'groups' => $groups,
            'platforms' => $platforms,
            'time_zone_list' => $time_zone_list,
            'customTemplates' => $customTemplates,
        ]);
    }

    public function update(Request $request, Campaign $campaign)
    {
          if(env('DEMO_MODE') && auth()->user()->id == 3){
            return back()->with('danger', __('Permission disabled for demo account please create a test account..!'));
        }
        return $this->store($request, $campaign);
    }

    public function destroy(Campaign $campaign)
    {
          if(env('DEMO_MODE') && auth()->user()->id == 3){
            return back()->with('danger', __('Permission disabled for demo account please create a test account..!'));
        }
        $campaign->delete();

        return back()->with('success', __('Deleted Successfully'));
    }

    public function send(Campaign $campaign)
    {
        CampaignService::send($campaign);

        return back()->with('success', 'Campaign Sent Successfully');
    }

    public function copy(Campaign $campaign)
    {
          if(env('DEMO_MODE') && auth()->user()->id == 3){
            return back()->with('danger', __('Permission disabled for demo account please create a test account..!'));
        }
        validateWorkspacePlan('campaign');

        $copy = $campaign->replicate();
        $copy->name = $copy->name.' - Copy';
        $copy->status = Campaign::STATUS_DRAFT;
        $copy->save();

        return to_route('user.whatsapp.campaigns.index')->with('success', 'Campaign Copied Successfully');
    }
}
